/*
 * MIT License
 *
 * Copyright (c) 2018 - 2025 CDAGaming (cstack2011@yahoo.com)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package com.gitlab.cdagaming.unilib.impl;

import com.gitlab.cdagaming.unilib.core.CoreUtils;
import com.mojang.blaze3d.platform.NativeImage;
import io.github.cdagaming.unicore.utils.FileUtils;
import io.github.cdagaming.unicore.utils.StringUtils;
import io.github.cdagaming.unicore.utils.TimeUtils;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.metadata.IIOMetadata;
import javax.imageio.metadata.IIOMetadataNode;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.WritableRaster;
import java.io.IOException;
import java.io.InputStream;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;

/**
 * Image Conversion Layers and Utilities used to translate other Image Types
 * <p>Reference: <a href="https://stackoverflow.com/a/17269591">Click Here</a>
 *
 * @author CDAGaming
 */
@SuppressWarnings("DuplicatedCode")
public class ImageFrame {
    /**
     * The delay between image transitions
     */
    private final int delay;
    /**
     * The buffered image instance being stored
     */
    private final BufferedImage image;
    /**
     * The native image instance being stored
     */
    private final NativeImage nativeImage;
    /**
     * The disposal method flag being used for this frame
     */
    private final String disposal;
    /**
     * The width of the current frame
     */
    private final int width;
    /**
     * The height of the current frame
     */
    private final int height;
    /**
     * The time, in milliseconds, for the frame to render until from current
     */
    private long renderTime = 0;

    /**
     * Initializes an Image Frame, with the specified arguments
     *
     * @param image    The buffered image, if any, to be stored for this frame
     * @param delay    The delay between now and the next image transition
     * @param disposal The disposal method flag to use for this frame
     * @param width    The width of this image
     * @param height   The height of this image
     */
    public ImageFrame(final BufferedImage image, final int delay, final String disposal, final int width, final int height) {
        this.image = deepCopy(image);
        this.nativeImage = null;
        this.delay = delay;
        this.disposal = disposal;
        this.width = width;
        this.height = height;
    }

    /**
     * Initializes an Image Frame, with the specified arguments
     *
     * @param image The buffered image, if any, to be stored for this frame
     */
    public ImageFrame(final BufferedImage image) {
        this(image, -1, null, -1, -1);
    }

    /**
     * Initializes an Image Frame, with the specified arguments
     *
     * @param image    The native image, if any, to be stored for this frame
     * @param delay    The delay between now and the next image transition
     * @param disposal The disposal method flag to use for this frame
     * @param width    The width of this image
     * @param height   The height of this image
     */
    public ImageFrame(final NativeImage image, final int delay, final String disposal, final int width, final int height) {
        this.image = null;
        this.nativeImage = image;
        this.delay = delay;
        this.disposal = disposal;
        this.width = width;
        this.height = height;
    }

    /**
     * Initializes an Image Frame, with the specified arguments
     *
     * @param image The native image, if any, to be stored for this frame
     */
    public ImageFrame(final NativeImage image) {
        this(image, -1, null, -1, -1);
    }

    /**
     * Initializes an Image Frame, with the specified arguments
     *
     * @param image       The buffered image, if any, to be stored for this frame
     * @param nativeImage The native image, if any, to be stored for this frame
     * @param delay       The delay between now and the next image transition
     * @param disposal    The disposal method flag to use for this frame
     * @param width       The width of this image
     * @param height      The height of this image
     */
    public ImageFrame(final BufferedImage image, final NativeImage nativeImage, final int delay, final String disposal, final int width, final int height) {
        this.image = deepCopy(image);
        this.nativeImage = nativeImage;
        this.delay = delay;
        this.disposal = disposal;
        this.width = width;
        this.height = height;
    }

    /**
     * Initializes an Image Frame, with the specified arguments
     *
     * @param image       The buffered image, if any, to be stored for this frame
     * @param nativeImage The native image, if any, to be stored for this frame
     */
    public ImageFrame(final BufferedImage image, final NativeImage nativeImage) {
        this(image, nativeImage, -1, null, -1, -1);
    }

    /**
     * Perform a deep-copy on the specified {@link BufferedImage}
     *
     * @param bi the target {@link BufferedImage}
     * @return the copied {@link BufferedImage}
     */
    public static BufferedImage deepCopy(final BufferedImage bi) {
        final ColorModel cm = bi.getColorModel();
        final boolean isAlphaPremultiplied = cm.isAlphaPremultiplied();
        final WritableRaster raster = bi.copyData(bi.getRaster().createCompatibleWritableRaster());
        return new BufferedImage(cm, raster, isAlphaPremultiplied, null).getSubimage(0, 0, bi.getWidth(), bi.getHeight());
    }

    /**
     * Returns Whether the inputted string matches the format of an external image type
     *
     * @param input The original string to parse
     * @return Whether the inputted string matches the format of an external image type
     */
    public static boolean isExternalImage(final String input) {
        return !StringUtils.isNullOrEmpty(input) &&
                (input.toLowerCase().startsWith("http") || StringUtils.isBase64(input).getFirst() || input.toLowerCase().startsWith("file://"));
    }

    /**
     * Decodes the inputted string into valid Base64 data if possible
     *
     * @param input             The string to parse data
     * @param encoding          The encoding to parse data in
     * @param useDecodingMethod Whether we're using the alternative decoding method
     * @param repeatCycle       Whether this is a repeat run with the same input, should be false except for internal usage
     * @return Valid Base64 data, if possible to convert string data
     */
    public static byte[] decodeBase64(final String input, final String encoding, final boolean useDecodingMethod, final boolean repeatCycle) {
        try {
            return Base64.getDecoder().decode(useDecodingMethod ? URLDecoder.decode(input, encoding) : input);
        } catch (Throwable ex) {
            CoreUtils.LOG.debugError(ex);

            if (!repeatCycle) {
                return decodeBase64(input, encoding, !useDecodingMethod, true);
            } else {
                return null;
            }
        }
    }

    /**
     * Reads an array of Image Frames from an InputStream
     *
     * @param stream The stream of data to be interpreted
     * @return The resulting array of Image Frames, if successful
     * @throws IOException If an error occurs during operation
     */
    public static ImageFrame[] readWebp(final InputStream stream) throws IOException {
        final ArrayList<ImageFrame> frames = new ArrayList<>(2);

        final ImageReader reader = ImageIO.getImageReadersByFormatName("webp").next();
        reader.setInput(ImageIO.createImageInputStream(stream));

        int width = -1;
        int height = -1;

        Color backgroundColor = null;

        BufferedImage master = null;
        boolean hasBackground = false;

        final int frameCount = reader.getNumImages(true); // Force reading of all frames

        final Class<?> animFrameClass = FileUtils.findClass("com.twelvemonkeys.imageio.plugins.webp.AnimationFrame");
        final List<?> frameData = (List<?>) StringUtils.getField(FileUtils.findClass("com.twelvemonkeys.imageio.plugins.webp.WebPImageReader"), reader, "frames");

        for (int frameIndex = 0; frameIndex < frameCount; frameIndex++) {
            final BufferedImage image = reader.read(frameIndex);
            final Object frameInfo = (frameData != null && !frameData.isEmpty() && frameIndex < frameData.size()) ? frameData.get(frameIndex) : null;

            if (width == -1 || height == -1) {
                width = image.getWidth();
                height = image.getHeight();
            }

            final int delay = frameInfo != null ? (int) StringUtils.getField(animFrameClass, frameInfo, "duration") / 10 : 0;
            final String disposal = "";

            if (master == null) {
                master = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);

                master.createGraphics().setColor(backgroundColor);
                master.createGraphics().fillRect(0, 0, master.getWidth(), master.getHeight());

                hasBackground = image.getWidth() == width && image.getHeight() == height;

                master.createGraphics().drawImage(image, 0, 0, null);
            } else {
                // WebP reader sometimes provides delta frames, (only the pixels that changed since the last frame)
                // so instead of overwriting the image every frame, we draw delta frames on top of the previous frame
                // to keep a complete image.
                master = deepCopy(frames.get(frameIndex - 1).getImage());

                if (frameInfo != null) {
                    final Rectangle bounds = (Rectangle) StringUtils.getField(animFrameClass, frameInfo, "bounds");
                    master.createGraphics().drawImage(image, bounds.x, bounds.y, null);
                } else {
                    master.createGraphics().drawImage(image, 0, 0, null);
                }
            }

            final BufferedImage copy = deepCopy(master);
            final NativeImage imgNew = convertToNative(copy, true);
            frames.add(new ImageFrame(copy, imgNew, delay, disposal, copy.getWidth(), copy.getHeight()));

            master.flush();
        }
        reader.dispose();

        return frames.toArray(new ImageFrame[0]);
    }

    /**
     * Reads an array of Image Frames from an InputStream
     *
     * @param stream The stream of data to be interpreted
     * @return The resulting array of Image Frames, if successful
     * @throws IOException If an error occurs during operation
     */
    public static ImageFrame[] readGif(final InputStream stream) throws IOException {
        final ArrayList<ImageFrame> frames = new ArrayList<>(2);

        final ImageReader reader = ImageIO.getImageReadersByFormatName("gif").next();
        reader.setInput(ImageIO.createImageInputStream(stream));

        int lastX = 0;
        int lastY = 0;

        int width = -1;
        int height = -1;

        final IIOMetadata metadata = reader.getStreamMetadata();

        Color backgroundColor = null;

        if (metadata != null) {
            final IIOMetadataNode globalRoot = (IIOMetadataNode) metadata.getAsTree(metadata.getNativeMetadataFormatName());

            final NodeList globalColorTable = globalRoot.getElementsByTagName("GlobalColorTable");
            final NodeList globalScreeDescriptor = globalRoot.getElementsByTagName("LogicalScreenDescriptor");

            if (globalScreeDescriptor.getLength() > 0) {
                final IIOMetadataNode screenDescriptor = (IIOMetadataNode) globalScreeDescriptor.item(0);

                if (screenDescriptor != null) {
                    width = Integer.parseInt(screenDescriptor.getAttribute("logicalScreenWidth"));
                    height = Integer.parseInt(screenDescriptor.getAttribute("logicalScreenHeight"));
                }
            }

            if (globalColorTable.getLength() > 0) {
                final IIOMetadataNode colorTable = (IIOMetadataNode) globalColorTable.item(0);

                if (colorTable != null) {
                    final String backgroundIndex = colorTable.getAttribute("backgroundColorIndex");

                    IIOMetadataNode colorEntry = (IIOMetadataNode) colorTable.getFirstChild();
                    while (colorEntry != null) {
                        if (colorEntry.getAttribute("index").equals(backgroundIndex)) {
                            final int red = Integer.parseInt(colorEntry.getAttribute("red"));
                            final int green = Integer.parseInt(colorEntry.getAttribute("green"));
                            final int blue = Integer.parseInt(colorEntry.getAttribute("blue"));

                            backgroundColor = StringUtils.getColorFrom(red, green, blue);
                            break;
                        }

                        colorEntry = (IIOMetadataNode) colorEntry.getNextSibling();
                    }
                }
            }
        }

        BufferedImage master = null;
        boolean hasBackground = false;

        for (int frameIndex = 0; ; frameIndex++) {
            final BufferedImage image;
            try {
                image = reader.read(frameIndex);
            } catch (IndexOutOfBoundsException io) {
                break;
            }

            if (width == -1 || height == -1) {
                width = image.getWidth();
                height = image.getHeight();
            }

            final IIOMetadataNode root = (IIOMetadataNode) reader.getImageMetadata(frameIndex).getAsTree("javax_imageio_gif_image_1.0");
            final IIOMetadataNode gce = (IIOMetadataNode) root.getElementsByTagName("GraphicControlExtension").item(0);
            final NodeList children = root.getChildNodes();

            final int delay = Integer.parseInt(gce.getAttribute("delayTime"));
            final String disposal = gce.getAttribute("disposalMethod");

            if (master == null) {
                master = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);

                master.createGraphics().setColor(backgroundColor);
                master.createGraphics().fillRect(0, 0, master.getWidth(), master.getHeight());

                hasBackground = image.getWidth() == width && image.getHeight() == height;

                master.createGraphics().drawImage(image, 0, 0, null);
            } else {
                int x = 0;
                int y = 0;

                for (int nodeIndex = 0; nodeIndex < children.getLength(); nodeIndex++) {
                    final Node nodeItem = children.item(nodeIndex);

                    if (nodeItem.getNodeName().equals("ImageDescriptor")) {
                        final NamedNodeMap map = nodeItem.getAttributes();

                        x = Integer.parseInt(map.getNamedItem("imageLeftPosition").getNodeValue());
                        y = Integer.parseInt(map.getNamedItem("imageTopPosition").getNodeValue());
                    }
                }

                if (disposal.equals("restoreToPrevious")) {
                    BufferedImage from = null;
                    for (int i = frameIndex - 1; i >= 0; i--) {
                        if (!frames.get(i).getDisposal().equals("restoreToPrevious") || frameIndex == 0) {
                            from = frames.get(i).getImage();
                            break;
                        }
                    }

                    if (from != null) {
                        master = deepCopy(from);
                    }
                } else if (disposal.equals("restoreToBackgroundColor") && backgroundColor != null && (!hasBackground || frameIndex > 1)) {
                    master.createGraphics().fillRect(lastX, lastY, frames.get(frameIndex - 1).getWidth(), frames.get(frameIndex - 1).getHeight());
                }
                master.createGraphics().drawImage(image, x, y, null);

                lastX = x;
                lastY = y;
            }

            final BufferedImage copy = deepCopy(master);
            final NativeImage imgNew = convertToNative(copy, true);
            frames.add(new ImageFrame(copy, imgNew, delay, disposal, copy.getWidth(), copy.getHeight()));

            master.flush();
        }
        reader.dispose();

        return frames.toArray(new ImageFrame[0]);
    }

    /**
     * Converts the specified BufferImage into a NativeImage
     * <p>For usage in Minecraft 1.13+
     *
     * @param img    The buffered image to parse and convert
     * @param useStb Whether this is a stb-type image
     * @return The converted NativeImage
     */
    public static NativeImage convertToNative(final BufferedImage img, final boolean useStb) {
        final NativeImage newImage = new NativeImage(img.getWidth(), img.getHeight(), useStb);
        for (int x = 0; x < img.getWidth(); x++) {
            for (int y = 0; y < img.getHeight(); y++) {
                int rgb = img.getRGB(x, y);
                final int alpha = (rgb >> 24) & 0xFF;
                final int red = (rgb >> 16) & 0xFF;
                final int green = (rgb >> 8) & 0xFF;
                final int blue = rgb & 0xFF;
                rgb = ((alpha & 0xFF) << 24) |
                        ((blue & 0xFF) << 16) |
                        ((green & 0xFF) << 8) |
                        ((red & 0xFF));
                newImage.setPixelRGBA(x, y, rgb);
            }
        }

        return newImage;
    }

    /**
     * Retrieves the current buffered image being stored
     *
     * @return The current buffered image being stored
     */
    public BufferedImage getImage() {
        return deepCopy(image);
    }

    /**
     * Retrieves the current native image being stored
     *
     * @return The current native image being stored
     */
    public NativeImage getNativeImage() {
        return nativeImage;
    }

    /**
     * Retrieves the delay between image transitions
     *
     * @return The delay between image transitions
     */
    public int getDelay() {
        return delay;
    }

    /**
     * Retrieves the disposal method flag being used for this frame
     *
     * @return The disposal method flag being used for this frame
     */
    public String getDisposal() {
        return disposal;
    }

    /**
     * Retrieves the width of the current frame
     *
     * @return The width of the current frame
     */
    public int getWidth() {
        return width;
    }

    /**
     * Retrieves the height of the current frame
     *
     * @return The height of the current frame
     */
    public int getHeight() {
        return height;
    }

    /**
     * Retrieves the time, in milliseconds, for the frame to render until from current
     *
     * @return The time, in milliseconds, for the frame to render until from current
     */
    public long getRenderTime() {
        return renderTime;
    }

    /**
     * Sets the time, in milliseconds, for the frame to render until from current
     *
     * @param renderTime The new timestamp to render until
     */
    public void setRenderTime(final long renderTime) {
        this.renderTime = renderTime;
    }

    /**
     * Sets the time, in milliseconds, for the frame to render until from current
     */
    public void setRenderTime() {
        setRenderTime(TimeUtils.toEpochMilli());
    }

    /**
     * Determine whether this frame has rendered up to or past the delay
     *
     * @return Whether this frame has rendered up to or past the delay
     */
    public boolean shouldRenderNext() {
        return TimeUtils.toEpochMilli() - getRenderTime() > getDelay() * 10L;
    }
}
