/*
 * MIT License
 *
 * Copyright (c) 2018 - 2025 CDAGaming (cstack2011@yahoo.com)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package com.gitlab.cdagaming.unilib.utils;

import io.github.cdagaming.unicore.utils.StringUtils;
import net.minecraft.client.Minecraft;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.storage.LevelData;

/**
 * Game-Related Utilities used to Parse World Data
 *
 * @author CDAGaming
 */
public class WorldUtils {
    /**
     * Retrieve the Game World Instance
     *
     * @param client the game client instance
     * @return the Game World Instance
     */
    public static Level getWorld(final Minecraft client) {
        return client != null ? client.level : null;
    }

    /**
     * Retrieve the Entity World Instance
     *
     * @param entity The entity to interpret
     * @return the Entity World Instance
     */
    public static Level getWorld(final Entity entity) {
        return entity != null ? entity.level : null;
    }

    /**
     * Retrieve the Game Player Instance
     *
     * @param client the game client instance
     * @return the Game Player Instance
     */
    public static Player getPlayer(final Minecraft client) {
        return client != null ? client.player : null;
    }

    /**
     * Retrieves the entities display name, derived from the original supplied name
     *
     * @param entity          The entity to interpret
     * @param stripFormatting Whether the resulting name should have its formatting stripped
     * @return The formatted entity display name to use
     */
    public static String getEntityName(final Entity entity, final boolean stripFormatting) {
        String result = "";
        if (entity != null) {
            result = StringUtils.getOrDefault(
                    entity.getDisplayName().getString(),
                    entity.getName().getString()
            );
        }

        if (stripFormatting) {
            result = StringUtils.stripAllFormatting(result);
        }
        return result;
    }

    /**
     * Retrieves the entities display name, derived from the original supplied name
     *
     * @param entity The entity to interpret
     * @return The formatted entity display name to use
     */
    public static String getEntityName(final Entity entity) {
        return getEntityName(entity, true);
    }

    /**
     * Retrieve the weather, utilizing the world
     *
     * @param world The world object to interpret
     * @return the current weather data
     */
    public static String getWeather(final Level world) {
        String name = "clear";
        if (world != null) {
            final LevelData info = world.getLevelData();
            if (info.isThundering()) {
                name = "thunder";
            } else if (info.isRaining()) {
                name = "rain";
            } else {
                name = "clear";
            }
        }
        return name;
    }

    /**
     * Retrieve the weather, utilizing the entity's world
     *
     * @param entity The entity to interpret
     * @return the current weather data
     */
    public static String getWeather(final Entity entity) {
        return getWeather(getWorld(entity));
    }
}
